package RePEc::Index::Collections;

=pod 

=head1 NAME

RePEc::Index::Collections -- the local metadata collections
configuration

=head1 SYNOPSIS

 use RePEc::Index::Collections;

 &RePEc::Index::Collections::read_configuration( 'collections' );

 my $collection = $RePEc::Index::COLLECTIONS{RePEc};

=head1 DESCRIPTION

RePEc::Index is capable of monitoring several metadata collections in
parallel.  Different metadata collections have different formats and
different identifiers and lie in different parts of the filesystem.

This module does a simple job: it can read a configuration file and
store that configuration in a global hash
(i.e. %RePEc::Index::COLLECTIONS).

In that hash each collection is an object of some collection class.
At the time of writing there's only one collection class:
RePEc::Index::Collection::RePEc.  Each type of metadata collection
must have its own collection class.

The collection class is an abstraction of the data format and its
respective parser.  

=cut


use Carp::Assert;

use strict;

# use vars qw( %COLLECTIONS @COLLECTION_IDS );
# %COLLECTIONS = ();
# @COLLECTION_IDS = ();

sub read_configuration {
  my $file = shift;

  @RePEc::Index::COLLECTION_IDS = ();

  open FILE, $file;

  while( <FILE> ) {
    next if /^#/;
    next if /^\s*$/;
    
    my ( $dir, $type, $prefix, $options ) = ( $_ =~ m/([^\s]+)\s+([^\s]+)\s+([^\s]+)(?:\s+([^\s]+))?/ );

    my $class = "RePEc::Index::Collection::$type";

#    print "Using $class\n";
    eval " use $class ;";
    if( $@ ) {
      die "Invalid collection type: $type ($@)";
    }


    my %options;

    # parse options as a list of comma separated statements
    for( $options ) {
      my @opt = split ',', $options;
      foreach ( @opt ) {
        my $par;
        my $val;

        if( $_ =~ /(.+)=(.+)/ ) {
          $par = $1;
          $val = $2;

        } else {
          $par = $_;
          $val = 1;
        }
        $options {$par} = $val;
      }
    }


#    print "Creating a collection: type: $type, dir: $dir, prefix: $prefix\n";

    my $index_dir = $RePEc::Index::Config::DATADIR . "/" . $prefix;

    my $collection = $class -> new( $dir, $type, $prefix, $index_dir, \%options );
    
    if( not $collection ) { 
      die "Can't create a collection of type $type, dir: $dir, prefix: $prefix";
    }


    $RePEc::Index::COLLECTIONS{$prefix} = $collection;
    push @RePEc::Index::COLLECTION_IDS, $prefix;
  }

  close FILE;
  return \%RePEc::Index::COLLECTIONS;
}




1;

__END__




