package ARDB::Table;

use strict;
use warnings;

our $VERSION = '0.01';

use Carp::Assert;

# Preloaded methods go here.

# constructor

sub new
 { 
  my $class      = shift;
  my $table_name = shift;
 
  my $self = 
   {
    name     => $table_name,
    create_statement_body => '',
    
    ##################################
    ### obsolete, but still using: ###
    ##################################
    fields   => {},
    fields_list => [],
    
   };
    
  bless $self, $class;
  return $self;
 }

sub add_field
 {
  my $self   = shift;

  my $name   = shift;
  my $type   = shift;

  $self -> {fields} -> {$name} = $type;
  push @{ $self->{fields_list} }, $name;

  $self -> {create_statement_body} .= "$name $type,\n";
 }

sub create_table_statement
 {
  my $self = shift;
  my $data = shift;

  $self -> {create_statement_body} .= "$data\n";

 }


sub store_values
 {
  my $self       = shift;
  my $sql_object = shift;
  my $data       = shift;

  my $sql_data;
  my @params = ();
  
  foreach ( sort keys %$data )
   {
    next
     unless ( $data -> { $_ } );
    $sql_data .= " $_=? , ";
    push ( @params, $data -> { $_ } );
   }
  $sql_data =~ s/(.*\s),\s/$1/;
  print 'insert into '. $self -> {'name'} . ' set ' . $sql_data;
  $sql_object -> prepare ('insert into '. $self -> {'name'} . ' set ' . $sql_data);
  $sql_object -> execute ( @params );
 }


sub perform_create
 {
  my $self       = shift;
  my $sql_object = shift;

  $self -> {create_statement_body} =~ s/\s*,\s*$//;
  my $creation_params = $self -> {create_statement_body} ;
  my $table_name = $self -> {name} ;
  $sql_object -> prepare ( "CREATE TABLE $table_name ( $creation_params )" );
  return $sql_object -> execute ();
 }

sub perform_delete
 {
  my $self       = shift;
  my $sql_object = shift;

  my $table_name = $self -> {name} ;
  $sql_object -> prepare ( "DROP TABLE $table_name" );
  $sql_object -> execute ();
 }

sub store_record
 {
  my $self       = shift;
  my $record     = shift;
  my $sql_object = shift;
  
  my $table_name = $self -> {name};

  my @fields;
  my @values;
  while ( my ($field, $value) = each %$record )
   {
     assert( $field !~ m/\-/ , "field name contains a dash! ($field)" );
    push @fields, $field;
    push @values, $value;
   }
  my $statement = "REPLACE $table_name SET " . join ', ', grep {$_ .= '=?'} @fields;
  $sql_object -> prepare ( $statement );
  $sql_object -> execute ( @values );
#  print "Stored to $table_name\n";
  #XXX check for SQL errors
 }

sub delete_records
 {
  my $self        = shift;
  my $column_name = shift;
  my $id          = shift;
  my $sql_object  = shift;

  my $table_name = $self -> {name};
  $sql_object -> prepare ( "DELETE FROM $table_name WHERE $column_name=?" );
  $sql_object -> execute ( $id );
 
 }



##################################################################
#              package ARDB::Table::Map
##################################################################


package ARDB::Table::Map;


sub new 
 { 
  my $class = shift;
  my $name  = shift;
 
  return undef
   unless $name;
  
  my $self = 
   {
    name     => $name,
    fields   => {},
   };
    
  bless $self, $class;
    
  return $self;
    
}


sub name
 {
  my $self = shift;
  return $self -> {name};
 }

sub fields
 {
  my $self = shift;
  return $self -> {fields};
 }


use Carp qw( &confess );

sub add_field
 {
  my $self = shift;
  my $name = shift || confess;
  my $list = shift || confess;

  # changes by Iku on 2003-05-22 19:22
  if( $list =~ /::/ ) {
    $self -> {fields} -> {$name} = $list;
  } else {
    $self -> {fields} -> {$name} = [ split /\s*,\s*/, $list ] ;
  }
}


sub add_mapping
 {
  my $self = shift;
  my $map  = shift;

  %{ $self -> {fields} } = ( %{ $self -> {fields} }, %{ $map -> {fields} } );
 }


sub produce_record
 {
  my $self = shift;
  my $record = shift;

  my $fields = $self -> {fields};
  
  my $result = {};

  foreach my $field_name ( sort keys %$fields )
   {
     # changes by Iku on 2003-05-22 19:22
     my $content = $fields->{$field_name}; 

     if( ref $content eq 'ARRAY' ) {
       $result -> { $field_name } =
	 join ' ', $record ->get_value ( @{ $content } );

     } else {
       no strict;
       my @val = &{ $content } ( $record ) ;
       if( @val ) {
	 $result -> { $field_name } = join( ' ', @val );
       } else {
	 $result -> { $field_name } = '';
       } 
     }

   }
  
  return $result;
  
 }

sub store
 {
  my $self     = shift;
  my $relation = shift;

  $self -> {sql} -> prepare ('insert into '. $self -> {table_name} . ' values ( ?, ?, ?, ? )');
  if ($self -> {sql} -> execute ( @$relation ))
   {
    return 1;
   }
  else
   {
    return undef;
   }
 }

sub remove
 {
  my $self     = shift;
  my $relation = shift;

  my @condition = (' subject = ? ', ' relation = ? ', '  object = ? ', '  source = ? ');
  
  my $remains = 3;
  
  for ( my $counter = 0; $counter <= $remains; $counter++ )
   {
    if ( not defined ( $relation -> [$counter] ) )
     {
      splice ( @$relation, $counter, 1 );
      splice ( @condition, $counter, 1 );
      $remains--;
      $counter--;
     }
   }

  my $res;

  if ( scalar @condition )
   {
    my $where_statement = join 'and', @condition;

    $res = $self -> {sql} -> 
      do (
	  'delete from '. $self -> {table_name} . ' where ' . $where_statement,
	  {},
	  @$relation 
	 );
   }
  return $res;

 }

sub fetch
 {
  my $self     = shift;
  my $relation = shift;

  my $sql_res;

  if ( not defined $relation )
   {
    $self -> {sql} -> prepare ('select * from '. $self -> {table_name} );
    $sql_res = $self -> {sql} -> execute ( );
   }
  else 
   {
    my @condition_p = (' subject = ? ', ' relation = ? ', '  object = ? ', '  source = ? ');

    my @conditions = ();
    my @values     = ();
    my $counter    =  0;
    
    foreach my $v ( @$relation ) {
      if ( defined $v ) {
	push @conditions, $condition_p[$counter];
	push @values, $v;
      }
      $counter ++;
    }

    my  $where_statement = join 'and', @conditions;

    $self -> {sql} -> prepare ( 
       'select * from '. $self -> {table_name} . ' where ' . $where_statement 
			      );
    $sql_res = $self -> {sql} -> execute ( @values );
   }

  my @result;

  while ( $sql_res -> {row} ) 
   {  

    my $subject  = $sql_res -> {'row'} -> {'subject'};
    my $relation = $sql_res -> {'row'} -> {'relation'};
    my $object   = $sql_res -> {'row'} -> {'object'};
    my $source   = $sql_res -> {'row'} -> {'source'};

    push ( @result, [$subject, $relation, $object, $source] );
    ###  iterates to the next record in the query's return set
    $sql_res -> next();
   }

  return @result;
 }


# Autoload methods go after =cut, and are processed by the autosplit program.

1;

__END__
# 

=pod

=head1 NAME

ARDB::Table - class to hold configuration data for SQL DB tables 

=head1 SYNOPSIS

  use ARDB::Table;

  # ARDB::Table::Map:

<field-attribute-mapping map-name="documents">
	<field-associations
		title="title"
		subject="title,abstract,keywords"
		authors="author/name"
		authors_emails="author/email"
		keywords="keywords"
		abstract="abstract"
		creation_date="creation-date"
		handle="handle"
		special_field="My::ARDB::special_field_filter" />

</field-attribute-mapping>


perl code:


my $template = shift;

my $record = {};

$record->{title} = $template->{'title'}[0];
$record->{subject} = $template->{'title'}[0] .
			' '. $template->{'abstract'}[0] . 
			' '. $template->{'keywords'}[0] ;

$record->{author_emails} = ... $template->{author/email} ???;

$record->{special_field} = &My::ARDB::special_field_filter( $template );

return $record;




# ARDB::Table:

sub store_record {
  my $self       = shift;
  my $sql_helper = shift;
  my $record     = shift;


  ### issue an SQL statement 
  ### which will insert the record into the table...
  ### and return success status


}




=head1 DESCRIPTION

ARDB::Configuration will create objects of this class.  An object will
be responsible for holding configuration data for an SQL table.  ARDB 
will manage those SQL tables. 

Each Table has a name, a list of field specifications and additional 
"raw SQL" lines for CREATE TABLE statements.  

Each field specification will have a field name (all field names must be unique), 
and an SQL type definition for that field.

May be, this class will also implement more than just storing configuration data.  
Later (!) it might do much more: perform table creation, deletion, and so on.

=head2 EXPORT

None by default.


=head1 AUTHOR

Ivan Baktcheev and Ivan Kurmanov

=head1 SEE ALSO

L<ARDB>, L<ARDB::Configuration>

=cut


