package ACIS::Web;

use strict;
use warnings;

use XML::LibXSLT;
use XML::LibXML;

use ACIS::Web::Config;
use ACIS::Web::Session;
use ACIS::Web::UserData;

use ACIS::Common;
use ACIS::Web::Services;

use CGI;
use CGI::Untaint;

use CGI::Carp qw( fatalsToBrowser set_message carpout );
use Carp qw( verbose );


use utf8;

BEGIN
 {
  set_message (\&ACIS::Common::critical_message);
 }; 

use Data::Dumper;

use sql_helper;

use ACIS::Data::DumpXML qw(dump_xml);

use Storable;

sub new
 {
  my $class   = shift;
  my $params  = {@_};
  my $home    = $params -> {'home'};
  
  my $config_file = $params -> {'config'} || '.config';
  my $screen_file = $params -> {'screen-config'} || 'screens.xml';
  
  #-   ,     
  
  $home = $ACIS::LocalConfig::local_path
   unless (defined $home);
  
  debug "creating ACIS::Web object in $home";
  
  my $self    =
   {
    'config-file' => $config_file,
    'screen-file' => $screen_file,
    'home'        => $home,
    'session'     => undef,
    'sessionfile' => undef,
    'variables'   => {},
   };
  
  bless $self, $class;
  
  debug 'loading configuration';
  
  $self -> config;
  
  debug "fetch request data";
  
  my $query = new CGI;
  
  my $base_url = $self -> {'config'} -> {'base-url'};
  
  my $unescaped_url = $ENV{REQUEST_URI};
  $unescaped_url =~ s/%(\w\w)/chr(hex($1))/eg;
  
  my $requested_url =
   "http://$ENV{HTTP_HOST}$unescaped_url";
   
  my ($acis_request) = ($requested_url =~ /^$base_url\/?(.*?)(?=\?|\/?$)/);
  
  $acis_request = ''
   unless defined $acis_request;
  
  debug "processing url: $requested_url, base url: $base_url, difference: $acis_request";
  
  my ($screen_name, $session_id) = split '!', $acis_request;
  
  my $template_set = $self -> {'config'} -> {'template-set'};
  my $shared = $self -> {'config'} -> {'static-base-dir'};
  my $static = $self -> {'config'} -> {'static-base-url'};
  
  my $paths =
   {
    'home'        => $home,
    'shared'      => $shared,
    'static'      => $static,
    'presenters'  => "$home/presentation/$template_set",
   };
  
  if( defined $session_id )
   {
    $paths->{session}     = "$home/sessions/$session_id";
    $paths->{unconfirmed} = "$home/unconfirmed/$session_id.xml";
   }

  $self -> {'paths'} = $paths;
  
  $params = scalar $query -> param;
  
  $self -> {'presenter-data'} = 
   {
    'system' => 
     {
      'config' =>
       {
        'base-url'        => $base_url,
        'site-name'       => $self -> config -> {'site-name'},
        'admin-email'     => $self -> config -> {'admin-email'},
        'debug'           => $ACIS::DEBUG,
       },
      'debug-messages'    => '',
     },
    'request' =>
     {
      'screen'            => $screen_name,
     },
    'response' =>
     {
      'data'    => $self -> {'variables'},
     },
   };
  
  my @params = $query -> Vars;
  
  debug Dumper {@params};
  
  if ($params)
   {
    $self -> {'presenter-data'} -> {'request'} ->
     {'form'} -> {'input'} = {@params};
   }
  
   
  $self -> {'request'} =
   {
    'CGI'        => $query,
    'params'     => $params,
    'screen'     => $screen_name,
    'session-id' => $session_id,
   };
  
  debug "processing request screen: $screen_name"
   if defined $screen_name;
  
  debug "proceed with session: $session_id"
   if defined $session_id;
  
  return $self;
 }


##############################################################
###       ###
##############################################################

sub error
 {
  my $self = shift;
  my $error_id = shift;

  $self -> {'presenter-data'} -> {'response'} -> {'error'} =
   $error_id;
 }

sub message
 {
  my $self = shift;
  my $message_id = shift;

  $self -> {'presenter-data'} -> {'response'} -> {'message'} =
   $message_id;
 }

sub success
 {
  my $self = shift;
  my $success = shift;

  $self -> {'presenter-data'} -> {'response'} -> {'success'} =
   $success;
 }


sub config
 {
  my $self = shift;
  
  return $self -> {'config'}
      if $self -> {'config'};
  
  my $home = $self -> {'home'};
  my $config_file = $home . '/' . $self -> {'config-file'};
  my $screen_file = $home . '/' . $self -> {'screen-file'};
  
  my $config_object;
  
  my $bin_config_file = $config_file . '.binary';
  
  debug "compare timestamps of the config file '$config_file', its binary dump '$bin_config_file' and module ACIS::Web::Config";

  my $configuration_package_file = $INC {'ACIS/Web/Config.pm'};
  
  my $package_change = -M $configuration_package_file;
  my $config_change  = -M $config_file;
  my $screen_change  = -M $screen_file;
  my $binary_change  = -M $bin_config_file
   if -f $bin_config_file;
  
  my $last_modified = $config_change;
  my $action = 'parse';
  
  if ($package_change < $config_change)
   {
    debug "module are newer than configuration";
    $last_modified = $package_change;
    $action = 'parse';
   }
  
  if ($screen_change < $last_modified)
   {
    debug "screen configuration changes are newer than configuration";
    $last_modified = $screen_change;
    $action = 'parse';
   }
  
  if ( (-f $bin_config_file) and ($binary_change < $last_modified) )
   {
    debug "configuration binary newer than new";
    $action = 'restore';
   }
  
  if ($action eq 'parse')
   {
    debug 'parsing configuration';
    
    $config_object = retrieve ACIS::Web::Config ($home, $config_file, $screen_file)
     || critical "ACIS::Web::Config did not work";

    store ( $config_object, $bin_config_file )
     or log_error ( "cannot write binary config to '$bin_config_file'" );
   }
  elsif ($action eq 'restore')
   {
    debug "loading configuration binary" ;
    $config_object = retrieve ( $bin_config_file );
   }
  
  critical 'something wrong in configuration'
   unless defined $config_object and $config_object;
  
  $self -> {'config'} = $config_object;
  
  foreach my $module (@{$config_object -> {'modules'}})
   {
    debug "try using $module";
    eval "use $module;";
    critical $@ if $@;
   }
  
  return $self -> {'config'};
 }

sub errors
 {
  my $self = shift;
  return $self -> {'errors'};
 }

sub paths
 {
  my $self = shift;
  return $self -> {'paths'};
 }

sub request
 {
  my $self = shift;
  return $self -> {'request'};
 }

sub variables
 {
  my $self = shift;
  return $self -> {'variables'};
 }

sub user_data
 {
  my $self = shift;
  my $path = shift;
  
  unless (defined $self -> {'user-data'})
   {
    unless ($path)
     {
      $path = $self -> paths -> {'user-data'};
    
      critical "authorization required"
	unless defined $path and $path;
     }

    use Carp::Assert;
    assert( $path !~ m!//! );
     
    my $user_data = load ACIS::Web::UserData ($path);
    $self -> {'user-data'} = $user_data;
   }
  
  return $self -> {'user-data'};
 }

#sub user_record
# {
#  my $self = shift;
#  my $id   = shift || 0;
#  
#  return $self -> user_data -> {'records'} -> [$id];
# }
 
sub session 
 {
  my $app = shift;
  return $app -> {session};
 }

sub sql_object
 {
  my $self = shift;
  
  return $self -> {'sql-object'}
   if defined $self -> {'sql-object'};

  sql_helper -> set_log_filename ( $self ->{home} . '/sql.log' );

  my $config = $self -> config;

  $self -> {'sql-object'} = sql_helper ->
   new( $config -> {'db-name'}, $config -> {'db-user'}, $config -> {'db-pass'} )
    or return undef;
  
  return $self -> {'sql-object'};
 }

sub handle_request
 {
  my $self = shift;
  
  #-   (ACIS::Web::Config),   
  #  .
  
  my $config = $self -> {'config'};
  
  my $screen_name = $self -> request -> {'screen'};
  
  $screen_name = 'index'
   unless ($screen_name);
  
  unless (defined $self -> config -> screen ($screen_name))
   {
    $screen_name = 'index';
    $self -> error ('screen-not-found');
   }
  
  $self -> {'presenter-data'} -> {'request'} -> {'screen'} = $screen_name;
  
  $self -> add_to_process_queue ($screen_name);
  $self -> set_presenter ($screen_name);
  
  #-         
  
  while (my $processor = $self -> next_processor)
   {
    debug "launch '$processor'";
    
    no strict;
    &$processor ($self);
   }
  
  debug "processors finished";

  my $session = $self -> session;
  
  my $session_id;

  if ($session and defined $session -> id)
   {
    $session -> save;
    $session_id = $session -> id;
    debug "session saved";
   }

  

   #critical "ha-ha";
  
#  if (defined $self -> {'user-data'} and $self -> {'user-data'})
#   {
#    $self -> user_data -> save;
#    debug 'user-data saved';
#   }
   
  
  my $location = $self->{'redirect-to'};
  
  if ($location)
   {
    $ACIS::DEBUGIMMEDIATELY
                 ? print "Location: <a href='$location'>$location</a>\n\n" :
                   print "Location: $location\n\n";
    return;
   }
  
  my $base_url = $self -> {'config'} -> {'base-url'};
  my $action   = "$base_url/$screen_name";

  if ($session_id)
   {
    $action .= "!$session_id";
   }
  
  # XXX clean the mess!

  # for compatibility with some older XSLT I could also write session
  # id into {session}, but I won't

  ### Debugging trick

  $self -> {'presenter-data'} -> {'system'} -> {'debug-messages'} = $ACIS::Common::LOGCONTENTS;
  
  my $vars_xml_dumped = dump_xml ($self -> {'presenter-data'});
  
  debug "content: \n$vars_xml_dumped\n";
  
  my $content = launch_presenter
   ($self -> {'presenter'} -> {'type'},
    $vars_xml_dumped,
    $self -> {'presenter'} -> {'file'});
    
  print '</pre>'
   if $ACIS::DEBUGIMMEDIATELY;
  
  my $encoding = $self -> config -> {'encoding'};
  
  $encoding = 'utf-8'
   unless ( defined $encoding and $encoding );
  
  print "Content-Type: text/html; charset=$encoding\n\n";
  print $content;
  
 }

sub add_to_process_queue
 {
  my $self = shift;
  my $screen = shift;

  debug "add '$screen' screen to processor queue";

  my $processors;
  
  if ($self -> request -> {'params'})
   {
    $processors = $self -> config -> screen ($screen) -> {'process-calls'};
   }
  else
   {
    $processors = $self -> config -> screen ($screen) -> {'init-calls'};
   }

  push @{$self -> {'processors'}}, @$processors;

 }

sub set_presenter
 {
  my $self = shift;
  my $screen = shift;

  assert( $self );
  assert( $screen );
  assert( $self->config->screen( $screen ) );
  
  $self -> {'presenter'} =
   $self -> config -> screen ($screen) -> {'presentation'};
   
 }
 
sub clear_process_queue
 {
  my $self = shift;
  
  debug 'requested for clearing processor queue, processed';
  
  $self -> {'processors'} = [];
 }
 
sub next_processor
 {
  my $self = shift;
  
  return shift @{ $self -> {'processors'} };
 }

sub redirect_to_screen
 {
  my $self   = shift;
  my $screen = shift;
  
  my $base_url = $self -> {'config'} -> {'base-url'};

  if ( $self->session ) {
    my $session_id = $self -> session -> id;
    $self -> {'redirect-to'} = "$base_url/$screen!$session_id";

  } else {
    $self -> {'redirect-to'} = "$base_url/$screen";

  }
  

 }



sub redirect
 {
  my $self = shift;
  my $url  = shift;
  
  $self -> {'redirect-to'} = $url;
 }


sub launch_presenter
 {
  my $type = shift;
  my $data = shift;
  my $file = shift;
  
  my $result;
  
  my $error;
  my $msg = '';
  
  debug "applying stylesheet when needed and generate page content";
  
  unless (-f $file)
   {
    debug "we can't load file stylesheet '$file'";
    $error = 'found';
   }
  
  if ($type eq 'xslt' and not $error)
   {
    my $parser = new XML::LibXML;
    my $xslt   = new XML::LibXSLT;

#    $xslt -> debug_callback( \&xslt_debug );

    $parser -> expand_entities (0);
    $parser -> load_ext_dtd (0);
    
    my $binary_xslt = $file . '.binary';
    
    my $stylesheet;

    # parsing stylesheet
    {
      my $style_doc = $parser -> parse_file ($file);
      $stylesheet   = $xslt  -> parse_stylesheet($style_doc);
    };


    # transformation
    {
      my $source = $parser -> parse_string ($data);
      my $results = $stylesheet -> transform($source);
      $result = $stylesheet -> output_string($results);
    };
    
    if ($@)
     {
      $error = 'transform';
      $msg = "<b>error text:</b>$@\n<br><pre><tt>$data</tt></pre>\n<br>";
      die $@;
     }
   }
  elsif ($type eq 'static' and not $error)
   {
    open HTML_FILE, $file;
    local $/ = undef;
    $result = <HTML_FILE>;
    close HTML_FILE;
   }

  if ($error 
      or $result eq ''
      or $result =~ m!body></body! )
   {
    $result = qq[
    <html>
     <head><title>stylesheet error</title></head>
     <body>
      <h2>stylesheet parse error</h2>

      <p>we cannot $error stylesheet '$file'.</p>

      <p>$msg</p>

     </body>
    </html>];
   }
  return $result;
 }


sub xslt_debug {
  my $msg = join ( ', ', @_ );
  debug "[XSLT] $msg";
}


 
1;

__END__

=head1 NAME

ACIS::Web

=head1 SYNOPSIS

 use ACIS::Web;
 my $acis = new ACIS::Web ('t/acis-home');
 
 $xml = dump_xml(@list)

=head1 DESCRIPTION

     - ACIS.


This library is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=cut
