package ReDIF::Writer;

##  Copyright (c) 2002,2003 Ivan Kurmanov.  All rights reserved.
##
##  This code is free software; you can redistribute it and/or modify it
##  under the same terms as Perl itself.

##  It is part of ReDIF-perl kit.


use strict;
use Carp::Assert;

use Exporter;

use vars qw( @ISA @EXPORT_OK );

@ISA= qw( Exporter );
@EXPORT_OK = qw( &stringify_template );

=pod

=head1 NAME

ReDIF::Writer - A tool to convert in-memory template representation
into a parseable text string

=head1 SYNOPSIS

 use ReDIF::Writer qw( stringify_template );

 # read a template from somewhere
 my $template = redif_get_next_template ();

 # tweak a template
 $template->{note} = [ "This template has gone through special processing" ];

 # write it out
 my ( $string, $problems ) = stringify_template( $template );
 print $string;

 foreach ( @$problems ) {
   my $level   = $_->[0];
   my $message = $_ ->[1];
 } 

=cut


use ReDIF::Parser;

use vars qw( $spec 
             $structure $happened $skip 
             $attr $prefix $context
             $attrorder );

$spec = $ReDIF::Parser::Options{redif_specification};

$attrorder = [ qw( 
                  title
                  name

                  author
                  editor
                  programming-language
                  creation-date
                  revision-date
                  abstract
                  file

                  primary
                  secondary 
                  tertiary

                  maintainer-email
                  maintainer-name
                  maintainer-phone
                  maintainer-fax

                  handle

                  ) ];

my $string;
my @problems;

sub initialize {
  $string   = '';
  @problems = ();
  $prefix   = '';
  $context  = 'top level';
}


sub problem ($$) {
  my $severity = shift;
  my $message  = shift;
  if ( $context ) { $message .= " at $context";  }
#  print "PROBLEM $severity $message\n";
  push @problems, [$severity, $message];
}

sub error ($)   { problem 3, shift; }
sub warning ($) { problem 2, shift; }

sub check ($$) {
  my $test = shift;
  my $warn = shift;
  if ( not $test ) {
    warning $warn;
  }
}

sub ensure ($$) {
  my $test = shift;
  my $err  = shift;
  if ( not $test ) {
    error $err;
  }
}

sub attribute ($$;$) {
  my $name = shift;
  my $value_list = shift;

  my $def   = $structure ->{$name};
  my $max;
  
  if ( $def 
       and defined $def ->{max} ) {
    $max = $def->{max};
  }

  foreach ( @$value_list ) {
    if ( $_ ) {
      $happened ->{$name}++;
      ### occurence max limit
      if ( $max and $happened->{$name} > $max ) {
        warning "attribute $name is repeated";
        last;
      }
      if ( $def 
           and $def ->{type} ) {
        my $type = $spec ->type( $def->{type} );
        ###  check the attribute value XXX
#        print "attribute $name of type $def->{type}\n";
##        print " ", join( ' ', keys %$def ), "\n";
#        $ok = 1;
        local( $context ) = "item $name at $context";
        local( $attr )    = $name;
        $_ = check_attribute_value( $_, $type, $def->{type} );
      }
      if ( $_ ) {
        $string .= "$prefix$name: $_\n";
      }
    }
  }
}


sub process_data ($) {
  my $data = shift;
  local( $attr );

  foreach $attr ( @$attrorder, sort keys %$data ) {

    if ( $skip -> {$attr} ) { next; }
    if ( $attr =~ /[A-Z]/ ) { next; }    # rule out upper-case attributes
    if ( $happened -> {$attr} ) { next; }
    
    my $v = $data->{$attr};
    if ( not $v ) { next; }
    
    if ( not ref $v or ref $v ne 'ARRAY' ) {
      warning "hash item '$attr' must contain an array to become an attribute";
      next;
    }

    if ( $structure->{SWITCHERS}->{$attr} ) {
      my $cluster_type = $structure->{SWITCHERS}->{$attr} ;
      my $cluster_str  = $spec->context( $cluster_type );

      my $count = 0;
      foreach ( @$v ) {
        if ( $_ ) {
          $happened -> {$attr} = 1;
          local( $context ) = $prefix ? "$context/$attr\[$count]" : "$attr\[$count]";
          stringify_cluster( $_, "$prefix$attr-", $cluster_str, $cluster_type );
        }
        $count ++;
      }

    } elsif ( $structure->{$attr} ) {
      attribute $attr, $v;

    } elsif ( $attr =~ /^x\-/ ) {
      attribute $attr, $v;
      
    } else {
      error "unknown attribute '$attr'";
    }
  }

}


sub check_required_attributes () {
  foreach my $attr ( keys %$structure ) {

    if ( $skip -> {$attr} ) { next; }
    if ( $attr =~ /[A-Z]/ ) { next; }    # rule out upper-case attributes

    my $def = $structure ->{$attr};

    if ( not ref $def ) {
      warn "strange type definition: $def ($attr)";
    }
    if ( $def -> {min} ) {
      if ( $def -> {switch} ) {
        $attr = $def -> {prefix};
        $attr = substr( $attr, 0, length( $attr )-1 );
      }
      ensure $happened ->{$attr}, "required attribute '$attr' is missing";
    }
  }
}



sub stringify_template {
  my $t = my $template = shift;

  initialize();

  my $ha = $t->{handle};
  ensure( ($ha and $ha->[0]), "handle is obligatory in template" );

  my $tt = $t->{'template-type'};
  ensure $tt, "template-type is obligatory in template";

  if ( scalar @$tt > 1 ) {
    $tt = [ $tt->[0] ];
    warning "you can't have two template-type attributes";
  }
  attribute "template-type", $tt;

  local $structure = $spec -> template_type_context( lc $tt->[0] );

  ensure $structure, "invalid template type: $tt"; 
  assert( $structure->{KEY} );

  local $happened = {};
  local $skip     = { 'template-type' => 1,
                      'handle'        => 1  
                    };

  process_data $template;

  attribute 'handle', $t->{handle}; 
  $skip -> {handle} = 0;

  check_required_attributes;

  return( $string, \@problems );
}


sub stringify_cluster {
  my $data    = shift;
  local $prefix  = shift;
  local $structure = shift;
  my $type    = shift;

  assert( $data, "Cluster $prefix ($type) can not be empty" );
  assert( ref( $data ) eq 'HASH', 
          "Cluster $prefix ($type) must be in hash form" );

  local $skip     = {};
  local $happened = {};

  my $key_attr = $structure->{KEY}; 
  assert( $key_attr, 
         "Key attribute not defined for cluster $prefix ($type)" );

  my $v = $data ->{$key_attr};
  attribute $key_attr, $v;
  ensure( ($v and $v->[0]), "cluster's key attribute '$key_attr' must have a value" );

  ensure( scalar @$v == 1,
          "key attribute '$key_attr' of cluster $type is repeated" );

  $skip -> {$key_attr} = 1;

  process_data $data;
  check_required_attributes;
}


use vars qw( @MESSAGES );

sub error_buf($) {
  my $m = shift;
  push @MESSAGES, $m;
}

sub check_attribute_value( $$$ ) {
  my $value = shift;
  my $typed = shift;
  my $typename = shift;

  @MESSAGES = ();

  my $original_value = $value;
  my $may_ignore = defined( $structure->{$attr} {min} ) 
    ? 0 : 1;
#  if ( $structure ->{$attr} {key} ) { $may_ignore = 0; }

  if ( length( $value ) 
       and defined $typed
       and scalar keys %$typed
     ) {
        
    ###  only if the attribute is a key attribute or a required attribute,
    ###  it's errorneous value will generate an error,
    ###  otherwise it will only generate a warning, 
    ###  and it will be ignored
    
    my $t;
    if ( defined ($t = $typed->{'check-regex'}) ) {
      if ( $value !~ /$t/ix ) {
        error_buf "invalid value '$value' of type $typename (regex)";
        undef $value;
      }
    }
    
    if ( defined $typed->{'check-eval'} ) {
            
      my $check_func_ref = $typed->{'compiled-check-eval'} ;
      $ReDIF::Parser::Special::value = $value;
      @ReDIF::Parser::Special::ERRORS   = ();
      @ReDIF::Parser::Special::WARNINGS = ();
      my $errors   = \@ReDIF::Parser::Special::ERRORS;
      my $warnings = \@ReDIF::Parser::Special::WARNINGS;
      
      my $result = &$check_func_ref;
      $value = $ReDIF::Parser::Special::value;

      ###  main questions: 
      ###    is there is a usable value?
      ###    what messages should be displayed to the rech user?
      ###    consequently, is the template still valid ?
        
      ###  what if we accept the following: 
      ###    bad value should generate an error in eval-check 
      ###    or return back a false value.
      ###    that will invalidate the template only if the attribute
      ###    can not be ignored.  
      ###    Otherwise the error messages should be converted 
      ###    to warnings.
      ###    if the evaluation returned true, but some errors as well,
      ###    and we can ignore, we convert errors to warning and
      ###    use the value
      ###    if the evaluation returned true, and some errors as well,
      ###    but we can't ignore, we issue an error and thus
      ###    invalidate the template

      if ( not $result 
           or ( $result and scalar @$errors ) ) {
        undef $value;
        if ( not scalar @$errors ) {
          error_buf "invalid value '$original_value' of type $typename (eval)";
        }
      }
                
      foreach my $er ( @$errors, @$warnings ) {
        push @MESSAGES,  $er;
      }
    }

    if ( defined $typed->{length} ) {
      my $actual = length( $original_value );
      my $spec   = $typed->{length};
    
      if ( defined $spec -> {top} 
           and $actual > $spec -> {top} ) {
        error_buf "value is too long (top limit: $spec->{top})";
        
      } elsif ( defined $spec -> {max} 
                and $actual > $spec -> {max} ) {
        error_buf "value is suspiciously long (max limit: $spec->{max})";
        
      } elsif ( defined $spec -> {min} 
                and $actual < $spec -> {min} ) {
        error_buf "value is suspiciously short (min limit: $spec->{min})";
        
      } elsif ( defined $spec -> {bottom} 
                and $actual < $spec -> {bottom} ) {
        error_buf "value is too short (bottom limit: $spec->{bottom})";
        
      }
    }

  }
  foreach ( @MESSAGES ) { 
    if ( $may_ignore ) {
      warning $_;
    } else {
      error $_;
    }
  }
  
  return $value;
}


1;
