package RePEc::Index::Storage;

use strict;
use Carp::Assert;

my $main_flags = 
    DB_INIT_LOCK 
  | DB_INIT_LOG 
  | DB_INIT_MPOOL 
  | DB_INIT_TXN 
#  | DB_THREAD
  ;


sub prepare_for_work {
  my $dir = shift;
  my $cat = shift || $::CATASTROPHIC_RECOVERY;  ### catastrophic

  if ( $cat ) { 
    return run_recovery( $dir, 1 );
  } else {
    if ( not run_recovery( $dir, 0 ) ) {
      return 0; # fail miserably
#      return run_recovery( $dir, 1 );
    } else { return 1; }
  }
}


sub run_recovery {
  my $dir = shift;
  my $catastrophic = shift;

  my $recover_flag = $catastrophic ? DB_RECOVER_FATAL : DB_RECOVER;

  my $env = new BerkeleyDB::Env 
      (
       -Home     => $dir, 
       -Flags    => DB_CREATE | $recover_flag | $main_flags,
       -Mode     => 0664,
       -Verbose  => 1,
       -Config   => { DB_TMP_DIR => '/tmp' },
      );        
}


sub prepare_db_env {
  my $dir = shift;

  if ( not $Env ) {
    $Env = new BerkeleyDB::Env 
      (
       -Home     => $dir, 
       -Flags    => DB_CREATE | $main_flags,
       -Mode     => 0664,
       -Verbose  => 1,
       -Config   => { DB_TMP_DIR => '/tmp' },
     )
        or die "Can't get my hands on the DB in $dir: $BerkeleyDB::Error ($!)";
  }
  assert( $Env );
}


sub start_transaction {

#  log_it( 3, "start_transaction()" );
  my $res;

  if ( not $Env ) {
    warn "Starting a transaction, while there's no environment still!";
  } else {
    $res = $Env -> txn_begin( undef, DB_TXN_NOWAIT );
#    $res = $Env -> txn_begin();
  }

#  log_it( 3, "start_transaction exit $res" );
  return $res;
}



sub commit_transaction {
  my $txn = shift;

#  log_it( 3, "commit_transaction( $txn )" );

  if ( $txn ) {
    $txn -> txn_commit ()
      and warn "Txn->commit returned positive!\n\$! = '$!', BerkeleyDB::Error = '$BerkeleyDB::Error'";
  } else {
    warn "Commiting a transaction, while there's no transaction!";

  }

  undef $txn;
  #  log_it( 3, "commit_transaction exit" );  
}





sub load_record_from_db_txn {
  my ( $txn, $file_name, $key ) = @_;
  
#  log_it( 3, "load_record_from_db_txn( $txn, $file_name, $key )" );

  my $hash = open_dbfile( $file_name, 'write' ) 
    or return undef;

  assert( $hash );

  my $db = tied %$hash;
#  log_it( 3, "load_record_from_db_txn: db opened ok $db" );

  assert( $db  );
  assert( $txn );

  $db ->Txn( $txn );

  my $val;
  my $status = $db ->db_get( $key, $val, DB_RMW );

  if ( not defined $val or $status ) {
    if ( $status != DB_NOTFOUND
         and $status != DB_KEYEMPTY ) {
      log_it 5, "db_get: $status";    
    }
  }


  my $res;
  if ( defined $val ) {
    eval { $res = thaw( $val ); };
    if ( $@ ) {
      warn "load_record_from_db_txn( $key, $db ): $@";
      undef $res;
    }
  }

#  { no warnings; 
#    log_it( 3, "load_record_from_db_txn exit $res" );
#  }
  return $res;
}


sub load_record_from_db_txn_readonly {
  my ( $txn, $file_name, $key ) = @_;
  
#  log_it( 3, "load_record_from_db_txn_readonly( $txn, $file_name, $key )" );

  my $hash = open_dbfile( $file_name, 'read' ) 
    or return undef;

  assert( $hash );

  my $db = tied %$hash;
#  log_it( 3, "load_record_from_db_txn_readonly: db opened ok $db" );

  assert( $db  );
#  assert( $txn );

  $db -> Txn();

  my $val;
  my $status = $db->db_get( $key, $val );

  if ( not defined $val or $status ) {
    #    my $notfound = ( $status eq DB_NOTFOUND ) ? "NOT FOUND" : '';
    #    my $keyempty = ( $status eq DB_KEYEMPTY ) ? "KEY EMPTY" : '';
    if ( $status == DB_NOTFOUND ) { return undef; }

    if ( $status != DB_NOTFOUND 
         and $status != DB_KEYEMPTY ) {
      log_it 5, "db_get(ro): $status";    
    }
  }

  my $res;
  if ( defined $val ) {
    eval { $res = thaw( $val ); };
    if ( $@ ) {
      warn "load_record_from_db_txn_readonly( $key, $db ): $@";
      undef $res;
    }
  }
#  log_it( 3, "load_record_from_db_txn_readonly exit" );

  return $res;
}


##############################################################################
#   sub   SAVE RECORD TO DB   ################################################
##############################################################################

sub save_record_to_db_txn {
  my ( $txn, $file_name, $key, $record ) = @_;

#  log_it( 3, "save_record_to_db_txn( $txn, $file_name, $key, ... )" );
    
  my $hash = open_dbfile( $file_name, 'write' ) 
    or return undef;

  my $db = tied %$hash;

#  log_it( 3, "save_record_to_db_txn: db opened ok $db" );

  assert( $db  );

  $db -> Txn( $txn );

  my $value = freeze( $record );
  my $status;

#  log_it( 3, "save_record_to_db_txn: before db_put( $key )" );

  if ( defined $value ) {
    $status = $db->db_put( $key, $value );

    if ( $status ) {
      error( "db_put: $status / $BerkeleyDB::Error / $!" );
      undef $!;
    }
  }

#  log_it( 3, "save_record_to_db_txn exit $res" );
  return $status;
}


###############################################################################

#sub get_dbfile_hash_ref {
#    my $file_name = $_[0];
#    my $Data = open_dbfile( $file_name, 'read' );
#    return $Data;
#}

###############################################################################

sub delete_record_from_db_txn {
  my ( $txn, $file_name, $key ) = @_;
  
  my $hash = open_dbfile( $file_name, 'write' ) 
    or return undef;

  my $db = tied %$hash;

  assert( $db  );
#  assert( $txn );

  $db -> Txn( $txn );

  my $status = $db -> db_del( $key );

}


###############



my %OPEN_FILES; 
my @FILES;  ### $#FILES = 100;
my $Open_Op = 1;

my %permissions = ( read  => DB_RDONLY | DB_AUTO_COMMIT 
#                    | DB_THREAD
                    ,write => DB_CREATE | DB_AUTO_COMMIT 
#                    | DB_THREAD 
                  );


sub open_dbfile {
    my ( $filename, $permission ) = @_;

    my $hash_ref;
    
    my $data = $OPEN_FILES{$filename} ;

    if ( defined $data ) {

        my $re_open = 1;
        my $file_info = $data;
        my $p = $file_info->{perm};

        if ( ( ( $permission eq 'read' ) and ( $p eq 'write' ) ) 
             or ( $permission eq $p ) ) {
            $re_open = 0;
        }

        if ( not $re_open ) {
          assert( ref( $file_info->{hash} ) eq 'HASH' );
          $hash_ref = $file_info->{hash};

        } else {

            untie %{ $file_info->{hash} };
            my $perm = $permissions{ $permission };
            my %data;

            my $db = tie( %data, 'BerkeleyDB::Hash', 
                 -Env => $Env,
                 -Filename => $filename, 
                 -Flags    => $perm,
                 -Txn => undef
                        )
              or error "Cannot open file $filename: $! $BerkeleyDB::Error\n"; 

            warning "re-opened $filename ($permission) $file_info->{hash}";

            $file_info->{db}   = $db;
            $file_info->{hash} = \%data;
            $file_info->{perm} = $permission;
            $hash_ref = \%data;
        }
        $file_info -> {last_used} = $Open_Op;
        
    }  else {

        ### open a new item
        my $file_info = {
                         last_used => $Open_Op,
                         filename  => $filename, 
                         perm      => $permission,
                         };
        my %data;
        my $perm = $permissions{ $permission };
        my $ok = 
            tie( %data, 'BerkeleyDB::Hash', 
                 -Env => $Env,
                 -Filename => $filename, 
                 -Flags    => $perm,
                 -Txn => undef
               );

        warning "open $filename, perm: $perm ($permission) ";

        if ( $ok ) {
          $file_info -> {db}   = $ok; 
          $hash_ref = $file_info -> {hash} = \%data; 
          unshift @FILES, $file_info;
          $OPEN_FILES{$filename} = $file_info;
          $Open_Op ++;

        } else {
          error "Cannot open file $filename: $! $BerkeleyDB::Error\n"; 
        }
    }

    if ( not $Txn and not ( $Open_Op % 5 ) ) {   ### if $Open_Op = 3n, n is integer

        ###  remove (untie) old open file items

        ###  'old' is used 8 times ago;  what 'time' is defined elsewhere
        my $old_mark = $Open_Op - 8;
        
        my $c = 0;
        my $f;
        while ( 1 ) {    ### now check every open data file
            $f = $FILES[$c];

            if ( $f->{last_used} < $old_mark ) {
                untie %{$f->{hash}};   ### ?
                delete $OPEN_FILES{$f->{filename}};
                splice ( @FILES, $c, 1 ) ;

            } else {
                $c ++;
            }

            last if ( $c > $#FILES );  ### break out if reached the array end
        }

    }

    if ( defined $hash_ref ) {
      assert( ref $hash_ref );
      assert( ref( $hash_ref ) eq 'HASH' );
    }

    return $hash_ref;
}



sub release_res {
  undef $Txn;
  %OPEN_FILES = ();
  @FILES = ();
  undef $Env;
}

END {
  release_res();
}




1;

