##############################################################################
# 
#  Events [.pm]: - simple event-messaging mechanizm
#

package Events;

###   RCS keywords:
###
###   $Author$
###   $Date$
###   $Id$
###   $Revision$
###   $Source$
###
###   $Log$
###   Revision 1.4  2001/05/26 15:46:58  ivan
###   fixed the subroutine redefined warning!
###
###   Revision 1.3  2000/09/07 21:26:48  ivan
###   Acknowledged use of Carp::Assert module; included it as a prerequisite in Makefile.PL and README; version change to 0.2
###
###   Revision 1.2  2000/08/24 23:02:43  ivan
###   just some additions and little correction of the POD documentation
###
###   Revision 1.1  2000/08/21 11:21:31  ivan
###   Moved Events.pm from lib/Devel to lib/
###
###   Revision 1.3  2000/08/21 11:18:55  ivan
###   changed back to Events (instead from Devel::Events)
###
###   Revision 1.2  2000/08/21 07:39:15  ivan
###   Events to Devel::Events change and minor pod documentation changes
###
###   Revision 1.3  1999/12/14 10:54:11  ivan
###     fixed a mask processing bug ('?')
###
###   Revision 1.2  1999/12/14 10:49:44  ivan
###     Legacy code pieces deleted.   
###     Some POD documentation added.  
###     Name matching fixed and extended with the '?' character.
###
###   Revision 1.1  1999/12/14 05:37:13  ivan
###   Initial revision
###
###

$VERSION = "0.2";


=head1 NAME

Events - simple user-defined Events module to assist in
component-style development with Perl.

Note that the SYNOPSIS part of this page is most up-to-date and most
relevant, while the rest of the page needs a number of corrections and
additions.  But it still may be useful explaining some of the concepts
and features.

=head1 SYNOPSIS

    use Events;      ###  Load the module


    # Next line defines an event.  You have to register event before
    # you can actually use it (execute and handle it).  The first
    # parameter is the event id.  Supplying a description is optional.

    Events -> register_event ( 'SOME::EVENT' , "Some description" );


    #  This line registers an event handler by a subroutine reference.
    #  When a relevant event happens, the subroutine some_handler will
    #  be executed with all event data as parameters.  SOME::* is a
    #  wildcard for all events whose id is started with SOME::

    Events -> register_event_handler ( 'SOME::*', \&some_handler );


    #  The following code line sets a handler to every possible event
    #  by a piece of code in Perl.  The code will be executed each
    #  time an event happens.  (But eval function will not be executed
    #  every time, that's a performance-wise trick.)

    Events -> register_event_handler ( '*', 
				       'print join ( " ", @_ ), "\n";'
				       );

    #  This time we set a handler by the package name and function
    #  name to be called.  It will be called (in case of an
    #  appropriate event) like:
    #
    #         CLASS -> method_function ( @_ ); 
    #
    #  where @_ will contain all the event data.  SOME::? wildcard
    #  will catch SOME::EVENT but will not catch SOME::BAD::ERROR.

    Events -> register_event_handler ( 'SOME::?', 
				       [ 'CLASS' , 'method_function' ] );


    #  And finally we create (execute) an event and pass some data to
    #  the handlers.

    Events -> SOME::EVENT ( @Data );


    #  Now a bit about how to handle events.  Here we define an pretty
    #  simple universal handler, which will print the event id and the
    #  list of all the rest of event data supplied.
 
    sub some_handler {

	###  Each time the event ID is sent as a first parameter to
	###  all of the event's handlers. 

	my $event_id = shift; 

	print "Event Id: $event_id\n";
	print "Event Data: [ ", join( ', ', @_ ) , " ]\n";
	                     
    }

    #   Please note that you can not de-register an event handler or
    #   an event.  Note also that you can register zero or more
    #   handlers to a single event or to a group of events.  The
    #   handlers will be activated sequentially [Bin the order of their
    #   registration.


=head1 DESCRIPTION

This module provides a simple tool for a wide choice of different
call-back type operations between Perl modules.  It has been designed
to facilate the component-style development of applications in Perl.
The idea is to allow creation kind-of plug-and-play components from
the perl-modules.  In other words, the aim is to allow more abstract
modules interface definition.

For example, imagine a module which does something (clever).  Once it
does that thing, it can create an event.  Depending on wether you have
plugged in other related modules (which expect and process the
relevant event(s)) or not, the programm will step into the next stage or
will not.

Or depending on what particular components have you plugged-in, the
algorithm will go one or other way: each component (independently of
the others) may set it's own way to react to an event or to ignore it.

=head2  EVENTS 

Each event has a name (an ID).  And it is the main and the only way to
identify an event.

Once an event is happened, it is being handled.  An event is handled
by calling handler functions or executing the handler code by other
means (those will be discussed below).

An event can carry some information -- e.g. event-related data -- for
the handlers to process.

The key point is that an event is (usually) created in one place
(e.g. module), while the handlers are (usually) set in other places
(modules).  So as long as modules stick to the same event names and
event data expected, they can be plugged into a single mutual
application.

Before events can happen, their ID name must be registered.  To
provide a handler for an event or for a group of events, registering
event handlers is done.  Installing event handlers is optional.

Before an event can happen, its id name must be defined or registered.
This is done with a call to the C<register_event> () method.  An event
id is a string which must obey the same rules as subroutine names in
Perl.

Often it is reasonable to work out the event names before a name
conflict happens.  There is only one 'global' namespace for the event
names and you are is responsible for their uniqueness between multiple
Events-enabled modules used (should this situation ever happen.)

For this reason I thought that the following ideas may help.

Generally of course the events may and shell be grouped by their
meaning and/or origin.  And this shall be expressed in the event
names.  I suggest to use upper case letters for the event names and to
use double colon ('::') to separate different parts of the names.

For example, a workable event id may be 'DB::QUERY::EXECUTE'.

Event ID must look like a valid subroutine indetifier for perl.
(Because it will actually be used to create a subrouitne.)

Events are created with Events' class method register_event:

   Events -> register_event ( EVENT_ID [, DESCRIPTION] );

=head2  HANDLERS

A handler generally is some reaction to an event.  Here one can set an
event handler by a number of ways.

A handler may be a piece of Perl code to execute (in a string), or it
may be a function reference or it may be an object reference and a
method name to invoke on the event.

There may several handlers be set for the same event independently
(for instance in different components).

They will be executed in the order as they have been set.

To set a handler, call register_event_handler method of the Events
class:

     Events -> register_event_handler ( EVENT_ID,  HANDLER );

HANDLER here is either a string value (assuming: perl code), a
subroutine reference or an array of object (or class) reference and a
method name (reference).

=head2 EVENT ID MASKS

An interesting (may be) feature is that one can set a handler to a
group of events by a mask. E.g. the handler set for event id "SOME::*"
will be called for all events, whose id starts with "SOME::".

Two characters have special meaning in event ids and used for setting
event-group handlers: '*' and '?'.

'*' means any number of any trailing name-characters - [\w:_]*

'?' means one or more name-character, exculding colons - [\w:]+

Setting handlers for the event groups is as easy as setting handlers
for single events.  Just use event ID mask instead of the event id.

=head2 INVOKING EVENTS

An event may be invoked at any point.  To do this, use just the following:

    Events -> EVENT_ID ( @EVENT_DATA );

e.g.,

    Events -> DATA::NEXT_LINE ( $line );

The trick is that each event is dynamically defined by this module as
a subroutine or, more correctly, as an Events class' method.  And its
handlers are just included into the subroutine's body as perl code.
When a new handler is set for a subroutine, the appropriate method is
redefined.  This is made possible with some simple use of the Perl eval
function.


=head1 EXAMPLES

 sub empty_handler {}  ### for a subroutine reference, used below

 sub CleverHandler {
    
    my $event = shift;
    print "The '$event' handler:\n"; 
    print "Arguments: [ " , join ( ', ', @_ ) , " ]\n";
    
 }

 ###  Invoke an Event 
 Events -> RARE::EVENT ( 0, 1, [] );

=cut

###   use Exporter;

use Carp::Assert ':NDEBUG';

## use Carp;
use strict qw( vars subs refs );

use vars qw( %events %handlers );

%events   = ();
%handlers = ();

sub import {

    while ( $_ = shift ) {
	if ( not ref $_ ) {
	    Events->register_event ( $_ );
	}
    }

}


##########################  REGISTER  EVENT  ################################
###==========================================================================
sub register_event { #   EVENT_ID, DESCRIPTION
###==========================================================================

    shift;
    my( $event_id, $description ) = @_;
    assert( $event_id ) if DEBUG;

    my $events = \%events;

    if( $events -> { $event_id } ) {
	# already registered 
	return 0;

    } else {  
	
	$events -> { $event_id } = {};
	$events -> { $event_id }->{description} = $description;

###	execute_event( "CEA:New:event", $event_id, $description );
	
	#### normalize event-handler relationships
	#### see next function's comments

	my $handlers = \%handlers;
	foreach ( keys %$handlers ) {

 	    if ( match( $_, $event_id ) ) {
		push @{ $events -> { $event_id }->{handlers} }, 
		@{$handlers->{ $_ }}; 
		
		my ( $handler ) = $handlers -> { $_ } -> [0];
	
		if (DEBUG) {
		    print "An event found its handler: ",
		    "<$event_id> \n", 
		    "handler: ",
		    (ref($handler) eq 'ARRAY') 
			? "$handler->[0]\-\>$handler->[1]" 
			    :  $handler 
				, "\n"
				;
		}
	    }
	}

	build_event ( $event_id );       #### DEFINE THE EVENT METHOD
	return 1;
    }
}

######################  REGISTER  EVENT  HANDLER  ###########################
###==========================================================================
sub register_event_handler {   # EVENT_ID, HANDLER
###==========================================================================
    shift;

    my( $event_id, $handler ) = @_;
    assert( $event_id and $handler ) if DEBUG;

    my $events = \%events;
    my $handlers = \%handlers;

    if ( $event_id =~ /\*|\?/ ) {   ## Event id is a mask ?

	## if EVENT_ID is a mask (contains an asterisk '*' character), 
	## then store the handler reference in the %$handlers hash, 

	if( not defined $handlers -> { $event_id } ) {
	    $handlers -> { $event_id } = [ $handler ];
	} else {
	    push @{$handlers -> { $event_id }}, $handler; 
	}

	## Now need to normalize event-handler relationships

	my $mask = $event_id;
	my ($_event_id);
	foreach $_event_id ( keys %$events ) {
	    if ( match( $mask,  $_event_id ) ) {
		push @{ $events -> { $_event_id }->{handlers} }, 
            		 @{ $handlers -> { $mask }}; 

		my $handler = $handlers -> { $mask } -> [0];

		if (DEBUG) {
		    print "A handler found its event by mask (<$event_id>): ",
		    "<$_event_id> \n", 
		    "handler: ",
		    (ref($handler) eq 'ARRAY') 
			? "$handler->[0]\-\>$handler->[1]" 
			    :  $handler 
				, "\n"
				;
		}

		build_event ( $_event_id );       #### DEFINE THE EVENT METHOD
	    }
	}

    } else {
	## if EVENT_ID is a normal (full) event name, (doesn't contain an '*')
	## then copy it to each 

	if( not exists( $events -> {$event_id} ) ) {
	    $events -> { $event_id } = {};
	}
	push @{ $events -> { $event_id }->{handlers} }, $handler; 

	if (DEBUG) {
	    print "A handler found its event: ",
	    "<$event_id> \n", 
	    "handler: ",
	    (ref($handler) eq 'ARRAY') 
		? "$handler->[0]\-\>$handler->[1]" 
		    :  $handler 
			, "\n"
			;
	}
	build_event ( $event_id );       #### DEFINE THE EVENT METHOD 
    }
    
}




my $EVENT_HEADER = '   my $result = 1; '. "\n";
my $EVENT_FOOTER = '   return( $result );'. "\n";

use vars '@REFS' ;
@REFS = ();


##########################   BUILD EVENT   ###################################

sub build_event {
    my $event = shift;

    my $events = \%events;

    assert ( $events->{$event} );

    my $event_data = $events->{$event};

    my $event_text = & build_event_header( $event ) ; 

    foreach my $handler ( @{$event_data->{handlers}} ) {
	$event_text .= "\n   " ;

	if ( ref( $handler ) eq 'CODE' ) {

	    push @Events::REFS, $handler;
	    my $no = $#Events::REFS;

#	    $event_text .= "\$code = \$Events::CODE_REFS[$no];\n" ;
#	    $event_text .= "   & \$code ( \@\_ ); ";

	    $event_text .= " \$result = \&{ \$Events::REFS[$no] } ( \@\_ ); ";

	} elsif ( not ref( $handler ) ) {

	    $event_text .= $handler; 

	} elsif ( ref( $handler ) eq 'ARRAY' ) {

	    my $obj = $handler->[0];
	    my $meth = $handler->[1];
	    my $OO = 0;                 ### OBJECT ORIENTED ?

	    if ( ref( $obj ) ) {
		push @Events::REFS, $obj;
		my $no = $#Events::REFS;
		$obj = '$Events::REFS[' . $no . ']';
		$OO = 1;
	    }

	    if ( ref( $meth ) ) {
		push @Events::REFS, $obj;
		my $no = $#Events::REFS;
		$meth = '$Events::REFS[' . $no . ']';
		$OO = 2;
	    }
	    
	    if ( $OO ) { 
		$event_text .= $obj . " -> " ;
		if ( $OO == 1 ) {
		    $event_text .= $meth . ' ( @_ );   ### OO  ';
		} else {
		    $event_text .= ' &{ ' . $meth . ' } ( @_ );   ### OO  ';
		}
		
	    } else {
		
		$event_text .= '$result = ' . "${obj} -> ${meth} ( \@\_ );";
	    }

	} else {	    
	    warn "[Events] Don't know how to use handler: $handler\n";
	}
	
    }

    $event_text .= "\n\n$EVENT_FOOTER}";

    if (DEBUG) {
	print "Just built the event <$event>: -------\n" 
	    , $event_text, "\n"  , '-'x40 , "\n";
    }
    {
	no strict;
	undef &$event;
	eval $event_text;
    }

}

########################   BUILD  EVENT  HEADER    ###########################

sub build_event_header {
    my $ev = shift;

    return ( "sub $ev { \n" . 
##   "   my \$event_id = \$\_[0] = '$ev';\n$EVENT_HEADER" );

   "   my \$event_id = '$ev';\n" . 
   '   shift ( @_ ); unshift @_, ' . "'$ev' ;\n".	     
   "$EVENT_HEADER" 

#   "   my \$event_id = '$ev';\n" . 
#   '   $_[0] = ' . "'$ev' ;\n".	      "$EVENT_HEADER" 
	     );

}



###############################   MATCH    #################################

###=========================================================================
sub match {
###=========================================================================

###### testing code for this function: below :-) 

    my $mask = shift;
    my $e_id = shift;
    
#    print "Match: '$mask' ";

    $mask =~ s/\*/\[\\w\:_\]\*/g;   ### [\w:_]*   --- '*'
    $mask =~ s/\?/\[\\w_\]\+/g;    ### [\w_]+     --- '?' 

    my $res = ( $e_id =~ m/^${mask}$/ );

###    print "match: $e_id \=\~ /^$mask\$/ ::: " , $res , "\n";

    return $res;
}

######
###### the following code should produce three 'ok's !
###### 

### $t1 = Events::match( "Task::New::*" , "New::Luck" );
### $t2 = Events::match( "City::Luck::*" , "City::Luck::People" );
### $t3 = Events::match( "City::?::People" , "City::Love::People" );
### print "test 1: ",  ($t1) ? " failed " : " ok " , "\n";
### print "test 2: ", (!$t2) ? " failed " : " ok " , "\n";
### print "test 3: ", (!$t3) ? " failed " : " ok " , "\n";

### $t4 = Events::match( "City::?::People" , "City::Love::A::People" );
### print "test 4: ", ($t4) ? " failed " : " ok " , "\n";

######



1;


__END__
