#!/usr/bin/perl

use warnings;
use strict;

use Getopt::Std;
use Carp::Assert;
use Data::Dumper;
use Socket;

use Events;

my $silent;
my $ri_home;
my $spath;
my $pid_filename;

use vars qw( %options @channels %kids );


sub log_line;

### main code starts here:


### parse options

getopts( "doc:hsf:H:", \%options );


if( $options{'h'} ) { # help
 HELP:
  print <<ENDEND;
This is RePEc-Index update control daemon.

Usage: $0 -f FILE [other options]

Starts a RI update service daemon.

Options: 

  -f filename 
       Gives a name of a socket-file to bind to and on which to listen
       to update requests.  Must be absolute path.

  -H dir
       RePEc-Index home directory to use.  Must be absolute path.

  -d 
       Become a daemon

  -o
       Print logging messages to the standard output (and standart error)
       instead of the default log file (daemon.log) in the RePEc-Index 
       home dir

  -c number 
       Maximum number of children processes (serving that many concurrent
       update requests)

  -t 
       Upon startup, run catastrophic recovery of the databases.  
       (Instead of just normal recovery that is executed usually.)

  -s 
       be silent or, at least, not be verbose

ENDEND
 exit;
}

if( $options{s} ) {   # silent
  $silent = 1;
}

if( $options{f} ) {   # socket filename
  $spath = $options{f};
}


if ( $options{H} ) { $ri_home = $options{H}; }

if ( not $spath ) { goto HELP; }

if ( $options{t} ) { no warnings; $::CATASTROPHIC_RECOVERY = 1; }


$pid_filename = $ri_home . '/daemon.pid';



my $log = $ri_home . '/daemon.log';
if ( $options{o} ) { undef $log; }

sub reopen_stdout {
  if ( $log ) {
    require IO::Handle;
    if ( open STDOUT, ">>:utf8", $log ) {
      open STDERR, ">&STDOUT";
      STDOUT ->autoflush(1);
      STDERR ->autoflush(1);
    } 
    else {
      warn "failed to open logfile $log";
    }
  } 
}

reopen_stdout();

### deamon life variables 

my $should_exit;
my $busy       ;
my $handler_term = $SIG{TERM};


### install signal handler
$SIG{TERM} = \&exit_signal_handler;

sub exit_signal_handler {

  log_line "TERM SIGNAL";
  if ( $handler_term ) {  &$handler_term( @_ );   }
  if ( $busy ) {
    $should_exit = 1;

  } else {
    $should_exit = 2;
    &go_exit(1);
  }
}


use Symbol;
use POSIX qw(:signal_h :errno_h :sys_wait_h);


sub REAPER {
  my $pid;

  while ( $pid = waitpid( -1, WNOHANG ) ) {
    if ( $pid 
         and exists $kids{$pid}
         and WIFEXITED($?) ) {
      ### this kid died
      my $chan = $kids{$pid};
      log_line "kid on channel $chan finished with $?";
      delete $kids{$pid};
      undef $channels[$chan];

    } elsif ( $pid == -1 ) { 
      last;

    } else {
      log_line "waitpid( -1, WNOHANG ) returned $pid ($?)\n";
    }
  }

#  if ( $! ) { log_line "Sys: $!"; undef $!; }

#  my $found;
#  foreach ( keys %kids ) {
#    $pid = waitpid( $_, WNOHANG );
#    if ( $pid == $_ 
#         and WIFEXITED($?) ) {
#      ### this kid died
#      my $chan = $kids{$_};
#      print "kid on channel $chan finished with $?\n";
#      delete $kids{$_};
#      undef $channels[$chan];
#      $found = 1;
#    } else {
#      print "waitpid( $_, WNOHANG ) returned $pid ($?)\n";
#    }
#  } 
#  if ( not $found ) {
#    $pid = waitpid( -1, WNOHANG );
#    print "kid with pid $pid finished ($?)\n";
#  }
#
 
  $SIG{CHLD} = \&REAPER;
}



sub HUNTSMAN { 
#  local( $SIG{CHLD} ) = 'IGNORE';
  kill 'INT' => keys %kids;
  $should_exit = 1;
  go_exit(1);
}

$SIG{INT}  = \&HUNTSMAN;
$SIG{TERM} = \&HUNTSMAN;

my $kids_limit = $options{c} || 5;



log_line "attempting start";


require RePEc::Index;
if ( not RePEc::Index::check_and_prepare_db() ) {
  log_line "a database problem -- can't go on";
  exit;
}


if ( $options{d} ) {
  ### Now become a daemon
  use Proc::Daemon;
  Proc::Daemon::init();
  reopen_stdout();
  $SIG{CHLD} = \&REAPER;
  $SIG{INT}  = \&HUNTSMAN;
  $SIG{TERM} = \&HUNTSMAN;
}

create_pid_file();


my $uaddr = sockaddr_un $spath;
my $proto = getprotobyname 'tcp';

socket Server, PF_UNIX, SOCK_STREAM, 0
 or die "\nunable to create socket: $!\n";

unlink $spath;
bind Server, $uaddr
 or die "\nunable to bind socket: $!\n";

listen Server, SOMAXCONN;



log_line "server started on '$spath'";


$should_exit = 0;


###   the main loop

log_line "waiting for connections";

LOOP_START: 
for ( ; my $paddr = accept( Client, Server ); close Client ) {

  $busy = 1;
  
  log_line "connection on '$spath'";

  $| = 1;
  
  my %request;

  while ( 1 ) {
    my $request = <Client>;
    $request =~ s/\n|\r//g;
    last
      unless $request;
    my ( $key, $value ) = ( $request =~ /^([A-Z]+)\s(.*)$/ );
    $request {$key} = $value;
  }
  
  my $source     = $request {SOURCE};
  my $collection = $request {COLLECTION};
  my $what       = $request {UPDATE};
  my $force      = $request {FORCE };

  my $channel;
  my $kid_pid;

  if ( scalar keys %kids >= $kids_limit ) { 
    log_line "Too many kids, have to wait till one finishes";
    while ( scalar keys %kids >= $kids_limit ) {
      sleep 10;
      if ( $should_exit ) {
        log_line "A request is lost because we have to exit.";
        last LOOP_START;
      }
    }
  } 
  
  # find an available channel
  $channel = 0;
  foreach ( @channels ) {
    if ( not $_ ) {
      last;
    }
    $channel++;
    die if $channel > $kids_limit;
  }
  
  $channels[$channel] = { request => \%request };

  if ( defined $channel ) {

    ### block INT signal, as suggested by recipe 17.12 in the Perl Cookbook

    my $sigset = POSIX::SigSet -> new( SIGINT, SIGTERM, SIGCHLD );
    sigprocmask( SIG_BLOCK, $sigset ) 
      or die "Can't block certain signals for fork: $!";

    my $kid_pid = fork;
    if ( not $kid_pid ) { 
      
      ## child

      $SIG{INT}  = 'DEFAULT';
      $SIG{TERM} = 'DEFAULT';
      # unblock signals
      sigprocmask( SIG_UNBLOCK, $sigset ) or die "can't unblock signal";

      close Client;
      close Server;
      process_request( \%request, $channel ); 
      exit;

    } elsif ( $kid_pid ) { 
      ### forked !

      $channels[$channel] ->{pid} = $kid_pid;
      $kids{$kid_pid} = $channel;

      sigprocmask( SIG_UNBLOCK, $sigset ) or die "can't unblock signal";

      {
        no warnings;
        log_line "request:\n",
                 "   source: $source\n",
                 "   collection: $collection\n",
                 "   update: $what ($force)\n",
                 " channel: $channel\n pid: $kid_pid";
      }
  

    } else { 
      warn "Can't fork";
    }

  }

  $busy = 0;
  last if $should_exit;
  log_line "waiting for further connections";
}

if ( not $should_exit ) { goto LOOP_START; }

&go_exit;


sub go_exit {
  my $explicit = shift;
  
  log_line "exiting, because ", 
    $should_exit ? "I have to" 
      : "it seems, there's nothing to do";

#  require Carp;
#  Carp::cluck( "because..." );

  while ( scalar keys %kids ) {
    if ( wait() == -1 ) {
      last;
    }
  }

  &clean_pid_file ();
  
  if ( $explicit ) {
    exit;  
  }
}


####  end of main code 
 
sub process_request {
  my $request    = shift;
  my $channel    = shift;
  
  close STDOUT;
  
  my $logfile = $ri_home . '/update_ch' . $channel . '.log';
  if ( open STDOUT, '>>:utf8', $logfile ) {} 
  else { 
    die "Can't open log $logfile"; 
  }
  
  close STDERR;
  if ( open STDERR, '>&STDOUT' ) {}
  else { 
    die "Can't open err log $logfile"; 
  }

  close STDIN;
  
  require IO::Handle;
  STDOUT ->autoflush(1);
  STDERR ->autoflush(1);

  
  require RePEc::Index;
  require RePEc::Index::Update;
  
  if ( not $silent ) {
    Events->register_event_handler( "*", \&print_event );
  }
  
  my $source     = $request ->{SOURCE};
  my $collection = $request ->{COLLECTION};
  my $what       = $request ->{UPDATE};
  my $force      = $request ->{FORCE };
  
  {
    no warnings;
    log_line
      " request:\n source: $source\n collection: $collection\n update: $what ($force)";
  }
  
  record_request_time();
  
  my @par = ( $collection );
  if ( $force ) {
    push @par, "TOO_OLD_IS", $force; 
  }
  
  my $result;
  eval {
    my $update = RePEc::Index::Update->new( @par );
    
    if ( not $update ) { next; }

    my $abort = sub { $update ->ABORT; };
    my $pause = sub { $update ->PAUSE; };
    my $conti = sub { $update ->CONTINUE; };

    $SIG{TERM} = $abort;
    $SIG{INT}  = $abort;
    $SIG{STOP} = $pause;
    $SIG{CONT} = $conti;
    
    $result = $update ->process_this( $what );
  };
  if ( $@ ) { 
    log_line "Error: $@";
  }
  
  my $took = get_elapsed_time();
  
  $busy = 0;
  
  log_line "processed $what in $collection", ($result ? "" : " (aborted)") ;
  log_line "time: $took";
  
  if ( not defined $result ) { $result = 0; }
  exit $result;
}




sub log_line {
  no warnings;
  print scalar ( localtime ), " ", @_, "\n";
}



sub print_event {
  my $event = shift;
#  my @params = @_;
  my @params;

  foreach my $p ( @_ ) {
    if ( ref $p ) { next; }
    if ( not defined $p ) { next; }
    push @params, $p;
  }

  $event =~ s/RePEc::Index::Update/U/g;
  $event =~ s/RePEc::Index:://g;
  $event =~ s/::/ /;
  

  if ( $event eq 'U RECORD' ) { return; }
#  if ( $event =~ m/U RECORD / ) { return; }

  print "$event: ", join( '|', @params ), "\n";
}






sub create_pid_file {
  my $file = $pid_filename;

  assert( $file );

  warn "pid file already exists, going to overwrite it" if -f $file;

  if ( open PID, '>', $file ) {
    print PID $$;
    close PID;
  }
}



sub clean_pid_file {
  my $file = $pid_filename;

  if ( $file and -f $file ) {
    unlink $file;
    log_line "killed $file";
  }
}


use vars qw( $TIME );
use Benchmark;

sub record_request_time { 
  $TIME = new Benchmark;
}


sub get_elapsed_time {
  my $now  = new Benchmark;
  my $diff = timediff( $now, $TIME );
  return timestr( $diff );
}
  

